# main.py
"""
Main script to run dynamic allocation simulations and plot results.
Uses a unified simulation function and a separate plotting module.
"""

import matplotlib.pyplot as plt # Still needed if main does any direct plotting or plt.show()
import numpy as np
from collections import defaultdict
import random
import math # For sqrt
import json # For dumping simulations

# Import simulation components
import config
from simulation import run_unified_simulation
from mechanisms import planner_algorithm1_dual_sgd, planner_second_price_auction
from mechanisms import planner_optimistic_ftrl, planner_ftrl
from utils import discretize, get_report_from_action

from plot import plot_multiple_results


# --- Main Execution ---
if __name__ == "__main__":
    eta_Alg1 = 1.0 / math.sqrt(config.T) if config.T > 0 else 0.1
    eta_Alg1 = max(0.001, min(1.0, eta_Alg1)) # Clip eta to a reasonable range
    print(f"Using eta = {eta_Alg1:.4f} for Alg1.")

    eta_Epoched = 5.0
    print(f"Using eta = {eta_Epoched:.4f} for Epoched algorithms.")

    all_results_data = {} # Using a dictionary to store results for plotting

    # # --- Scenario 1: Q-learning Agents (adapting to mu) vs Algorithm 1 Planner ---
    name_s1 = "Q-Learning vs Primal-Dual"
    print(f"Starting Scenario 1: {name_s1}...")
    all_results_data[name_s1] = run_unified_simulation(
        planner_func=planner_algorithm1_dual_sgd,
        agent_type='q_learning',
        mechanism_name=name_s1,
        eta=eta_Alg1
    )
    print(f"Scenario {name_s1} finished.")

    # # --- Scenario 2: Q-learning Agents vs Optimistic FTRL (Numerical) Planner ---
    name_s2 = "Q-Learning vs our O-FTRL-FP"
    print(f"\nStarting Scenario 2: {name_s2}...")
    all_results_data[name_s2] = run_unified_simulation(
        planner_func=planner_optimistic_ftrl,
        agent_type='q_learning',
        mechanism_name=name_s2,
        eta=eta_Epoched
    )
    print(f"Scenario {name_s2} finished.")

    # # --- Scenario 3: Q-learning Agents vs FTRL Planner ---
    name_s3 = "Q-Learning vs our FTRL"
    print(f"\nStarting Scenario 3: {name_s3}...")
    all_results_data[name_s3] = run_unified_simulation(
        planner_func=planner_ftrl,
        agent_type='q_learning',
        mechanism_name=name_s3,
        eta=eta_Epoched
    )
    print(f"Scenario {name_s3} finished.")

    # # --- Scenario 4: Truthful Agents vs Algorithm 1 Planner ---
    name_s4 = "Truthful vs Alg1"
    print(f"\nStarting Scenario 4: {name_s4}...")
    all_results_data[name_s4] = run_unified_simulation(
        planner_func=planner_algorithm1_dual_sgd,
        agent_type='truthful',
        mechanism_name=name_s4,
        eta=eta_Alg1
    )
    print(f"Scenario {name_s4} finished.")

    # --- Plot Comparison ---
    print("\nGenerating comparison plots for all scenarios...")
    plot_multiple_results(all_results_data, name_s4)

